<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Commission;
use Illuminate\Support\Facades\DB;

class CleanupOrphanedCommissions extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'commissions:cleanup
                          {--dry-run : Display orphaned commissions without deleting them}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Cleanup orphaned commissions that have no associated completion';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('Checking for orphaned commissions...');

        // Find orphaned commissions (commissions without completions)
        $orphanedCommissions = Commission::whereNotExists(function ($query) {
            $query->select(DB::raw(1))
                ->from('completions')
                ->whereColumn('completions.completions_id', 'commissions.completions_id');
        })->get();

        $count = $orphanedCommissions->count();

        if ($count === 0) {
            $this->info('✓ No orphaned commissions found. Database is clean!');
            return self::SUCCESS;
        }

        $this->warn("Found {$count} orphaned commission(s):");

        // Display orphaned commissions
        $this->table(
            ['Commission ID', 'User ID', 'Completion ID', 'Amount', 'Created At'],
            $orphanedCommissions->map(function ($commission) {
                return [
                    $commission->commission_id,
                    $commission->user_id,
                    $commission->completions_id,
                    $commission->commission_amount,
                    $commission->created_at,
                ];
            })
        );

        if ($this->option('dry-run')) {
            $this->info('Dry run mode - no records deleted.');
            return self::SUCCESS;
        }

        if ($this->confirm('Do you want to delete these orphaned commissions?', true)) {
            $deleted = Commission::whereNotExists(function ($query) {
                $query->select(DB::raw(1))
                    ->from('completions')
                    ->whereColumn('completions.completions_id', 'commissions.completions_id');
            })->delete();

            $this->info("✓ Successfully deleted {$deleted} orphaned commission(s).");
            return self::SUCCESS;
        }

        $this->info('Cleanup cancelled.');
        return self::SUCCESS;
    }
}
