<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;

class TestCommissionApi extends Command
{
    protected $signature = 'test:commission-api';
    protected $description = 'Test commission API returns correct data (0 when empty)';

    public function handle()
    {
        $this->info('🧪 Testing Commission API');
        $this->newLine();

        // Check database
        $dbCount = DB::table('commissions')->count();
        $dbTotal = DB::table('commissions')->sum('commission_amount') ?? 0;

        $this->info("📊 Database Check:");
        $this->info("   Total commissions in DB: $dbCount");
        $this->info("   Total amount in DB: $$dbTotal");
        $this->newLine();

        // Test API endpoint (if server is running)
        $baseUrl = env('APP_URL', 'http://localhost:8000');

        try {
            $response = Http::get("$baseUrl/api/v1/commissions");

            if ($response->successful()) {
                $data = $response->json();

                $this->info("✅ API Response:");
                $this->info("   Success: " . ($data['success'] ? 'true' : 'false'));
                $this->info("   Count: " . count($data['data']));
                $this->info("   Total: $" . $data['total']);
                $this->newLine();

                // Check cache headers
                $cacheControl = $response->header('Cache-Control');
                $pragma = $response->header('Pragma');
                $expires = $response->header('Expires');

                $this->info("📋 Cache Headers:");
                $this->info("   Cache-Control: " . ($cacheControl ?? 'Not set'));
                $this->info("   Pragma: " . ($pragma ?? 'Not set'));
                $this->info("   Expires: " . ($expires ?? 'Not set'));
                $this->newLine();

                // Verify data matches database
                if ($data['total'] == $dbTotal && count($data['data']) == $dbCount) {
                    $this->info("✅ API data matches database!");
                } else {
                    $this->error("❌ API data DOES NOT match database!");
                    $this->error("   Expected: count=$dbCount, total=$$dbTotal");
                    $this->error("   Got: count=" . count($data['data']) . ", total=$" . $data['total']);
                }
            } else {
                $this->error("❌ API request failed with status: " . $response->status());
            }
        } catch (\Exception $e) {
            $this->warn("⚠️  Could not test API endpoint (server may not be running)");
            $this->warn("   Error: " . $e->getMessage());
        }

        $this->newLine();
        $this->info("💡 Tip: Make sure to:");
        $this->info("   1. Hard refresh the frontend (Ctrl+Shift+R or Cmd+Shift+R)");
        $this->info("   2. Clear browser cache");
        $this->info("   3. Check Network tab in DevTools for Cache-Control headers");

        return 0;
    }
}
