<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class TestDatabaseTriggers extends Command
{
    protected $signature = 'test:database-triggers';
    protected $description = 'Test PostgreSQL triggers by making direct database changes';

    public function handle()
    {
        $this->info('🧪 Testing PostgreSQL Triggers - Direct Database Changes');
        $this->newLine();

        DB::beginTransaction();

        try {
            // Get test data
            $user = DB::table('users')->first();
            $service = DB::table('services')->first();

            if (!$user || !$service) {
                $this->error('❌ Need at least one User and one Service in database');
                DB::rollBack();
                return 1;
            }

            // TEST 1: Insert approved completion using raw SQL
            $this->info('📝 Test 1: INSERT approved completion via RAW SQL → Commission created');

            DB::statement("
                INSERT INTO completions (service_id, user_id, charged, approved, date, created_at)
                VALUES (?, ?, ?, ?, ?, ?)
            ", [$service->service_id, $user->user_id, 100, true, now(), now()]);

            $completionId1 = DB::getPdo()->lastInsertId();

            $commission1 = DB::table('commissions')->where('completions_id', $completionId1)->first();
            $expectedAmount = (100 * $service->commission) / 100;

            if ($commission1 && abs($commission1->commission_amount - $expectedAmount) < 0.01) {
                $this->info("   ✅ Trigger created commission: $" . $commission1->commission_amount);
            } else {
                $this->error("   ❌ Commission not created by trigger");
            }
            $this->newLine();

            // TEST 2: Update approved from true to false using raw SQL
            $this->info('📝 Test 2: UPDATE approved TRUE → FALSE via RAW SQL → Commission deleted');
            $this->info("   Commission ID before: {$commission1->commission_id}");

            DB::table('completions')
                ->where('completions_id', $completionId1)
                ->update(['approved' => false]);

            $commission1After = DB::table('commissions')->where('completions_id', $completionId1)->first();

            if (!$commission1After) {
                $this->info('   ✅ Trigger deleted commission when unapproved');
            } else {
                $this->error('   ❌ Commission still exists after unapproval');
            }
            $this->newLine();

            // TEST 3: Update approved from false to true using raw SQL
            $this->info('📝 Test 3: UPDATE approved FALSE → TRUE via RAW SQL → Commission created');

            DB::table('completions')
                ->where('completions_id', $completionId1)
                ->update(['approved' => true]);

            $commission1New = DB::table('commissions')->where('completions_id', $completionId1)->first();

            if ($commission1New) {
                $this->info("   ✅ Trigger re-created commission: ID {$commission1New->commission_id}, Amount: $" . $commission1New->commission_amount);
            } else {
                $this->error('   ❌ Commission not created when re-approved');
            }
            $this->newLine();

            // TEST 4: Update charged amount using raw SQL
            $this->info('📝 Test 4: UPDATE charged (100 → 250) via RAW SQL → Commission updated');
            $oldAmount = $commission1New->commission_amount;
            $this->info("   Old commission: $$oldAmount");

            DB::table('completions')
                ->where('completions_id', $completionId1)
                ->update(['charged' => 250]);

            $commission1Updated = DB::table('commissions')->where('completions_id', $completionId1)->first();
            $expectedAmount4 = (250 * $service->commission) / 100;

            if ($commission1Updated && abs($commission1Updated->commission_amount - $expectedAmount4) < 0.01) {
                $this->info("   ✅ Trigger updated commission to: $" . $commission1Updated->commission_amount);
            } else {
                $this->error('   ❌ Commission not updated by trigger');
            }
            $this->newLine();

            // TEST 5: Change service using raw SQL
            if (DB::table('services')->count() > 1) {
                $this->info('📝 Test 5: UPDATE service via RAW SQL → Commission recalculated');
                $service2 = DB::table('services')->where('service_id', '!=', $service->service_id)->first();
                $oldRate = $service->commission;
                $newRate = $service2->commission;
                $this->info("   Service rate: $oldRate% → $newRate%");

                DB::table('completions')
                    ->where('completions_id', $completionId1)
                    ->update(['service_id' => $service2->service_id]);

                $commission1AfterService = DB::table('commissions')->where('completions_id', $completionId1)->first();
                $expectedAmount5 = (250 * $service2->commission) / 100;

                if ($commission1AfterService && abs($commission1AfterService->commission_amount - $expectedAmount5) < 0.01) {
                    $this->info("   ✅ Trigger recalculated commission: $" . $commission1AfterService->commission_amount);
                } else {
                    $this->error('   ❌ Commission not recalculated by trigger');
                }
                $this->newLine();
            }

            // TEST 6: Delete completion using raw SQL
            $this->info('📝 Test 6: DELETE completion via RAW SQL → Commission deleted');
            $commissionIdBefore = DB::table('commissions')->where('completions_id', $completionId1)->value('commission_id');
            $this->info("   Commission ID before delete: $commissionIdBefore");

            DB::table('completions')->where('completions_id', $completionId1)->delete();

            $commissionAfterDelete = DB::table('commissions')->find($commissionIdBefore);

            if (!$commissionAfterDelete) {
                $this->info('   ✅ Trigger deleted commission with completion');
            } else {
                $this->error('   ❌ Commission still exists after completion deleted');
            }
            $this->newLine();

            DB::rollBack();
            $this->info('✅ All database trigger tests completed! (Changes rolled back)');
            $this->newLine();
            $this->info('📊 Summary:');
            $this->info('   - PostgreSQL triggers work with DIRECT database changes');
            $this->info('   - Works with: SQL UPDATE, INSERT, DELETE commands');
            $this->info('   - Works with: Database tools (pgAdmin, DBeaver, etc.)');
            $this->info('   - Works with: Raw queries, manual edits, etc.');
            $this->info('   - Laravel Observers ALSO still work for Eloquent changes');

            return 0;
        } catch (\Exception $e) {
            DB::rollBack();
            $this->error('❌ Test failed: ' . $e->getMessage());
            $this->error($e->getTraceAsString());
            return 1;
        }
    }
}
