<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Service;
use App\Models\Completion;
use App\Models\Commission;
use Carbon\Carbon;

class CompletionsSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get some users and services to work with
        $users = User::where('role', 'employee')->pluck('user_id')->toArray();
        $services = Service::all();

        if (empty($users) || $services->isEmpty()) {
            $this->command->warn('No users or services found. Please seed users and services first.');
            return;
        }

        // Sample completions data
        $completions = [
            // Recent completions from today
            [
                'service_id' => $services->where('service_name', 'Haircut')->first()->service_id ?? $services->first()->service_id,
                'user_id' => $users[0],
                'charged' => 200.00,
                'date' => Carbon::now(),
                'notes' => 'Cash payment',
                'approved' => true,
            ],
            [
                'service_id' => $services->where('service_name', 'Manicure')->first()->service_id ?? $services->first()->service_id,
                'user_id' => $users[1] ?? $users[0],
                'charged' => 500.00,
                'date' => Carbon::now(),
                'notes' => 'Mpesa payment',
                'approved' => true,
            ],
            [
                'service_id' => $services->where('service_name', 'Pedicure')->first()->service_id ?? $services->first()->service_id,
                'user_id' => $users[1] ?? $users[0],
                'charged' => 700.00,
                'date' => Carbon::now(),
                'notes' => 'Card payment',
                'approved' => true,
            ],
            [
                'service_id' => $services->where('service_name', 'braiding')->first()->service_id ?? $services->first()->service_id,
                'user_id' => $users[2] ?? $users[0],
                'charged' => 1000.00,
                'date' => Carbon::now(),
                'notes' => 'Cash',
                'approved' => true,
            ],
            [
                'service_id' => $services->where('service_name', 'Beards')->first()->service_id ?? $services->first()->service_id,
                'user_id' => $users[3] ?? $users[0],
                'charged' => 100.00,
                'date' => Carbon::now(),
                'notes' => 'Mpesa',
                'approved' => true,
            ],
            // Yesterday's completions
            [
                'service_id' => $services->where('service_name', 'Facial')->first()->service_id ?? $services->first()->service_id,
                'user_id' => $users[0],
                'charged' => 1200.00,
                'date' => Carbon::yesterday(),
                'notes' => 'Customer satisfied',
                'approved' => true,
            ],
            [
                'service_id' => $services->where('service_name', 'Retouch')->first()->service_id ?? $services->first()->service_id,
                'user_id' => $users[1] ?? $users[0],
                'charged' => 800.00,
                'date' => Carbon::yesterday(),
                'notes' => 'Regular customer',
                'approved' => true,
            ],
            [
                'service_id' => $services->where('service_name', 'Hair Color')->first()->service_id ?? $services->first()->service_id,
                'user_id' => $users[2] ?? $users[0],
                'charged' => 500.00,
                'date' => Carbon::yesterday(),
                'notes' => null,
                'approved' => true,
            ],
            // Pending approval (won't trigger commission yet)
            [
                'service_id' => $services->where('service_name', 'Lashes')->first()->service_id ?? $services->first()->service_id,
                'user_id' => $users[0],
                'charged' => 800.00,
                'date' => Carbon::now(),
                'notes' => 'Pending approval',
                'approved' => false,
            ],
        ];

        $this->command->info('Inserting completions...');

        foreach ($completions as $completionData) {
            $completion = Completion::create($completionData);

            // Only create commission for approved completions
            if ($completionData['approved']) {
                // Insert commission record - trigger will auto-calculate the amount
                Commission::create([
                    'user_id' => $completionData['user_id'],
                    'completions_id' => $completion->completions_id,
                    'commission_amount' => 0, // Trigger will calculate this
                ]);
            }
        }

        $this->command->info('Completions seeded successfully!');
    }
}
