<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class SalesSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $inventoryItems = DB::table('inventory_items')->where('quantity', '>', 0)->get();
        $users = DB::table('users')->pluck('user_id')->toArray();
        
        if ($inventoryItems->isEmpty()) {
            $this->command->warn('No inventory items found. Please run InventoryItemsSeeder first.');
            return;
        }
        
        if (empty($users)) {
            $this->command->warn('No users found. Please run UsersSeeder first.');
            return;
        }

        // Create 10 sample sales
        for ($i = 1; $i <= 10; $i++) {
            $saleDate = now()->subDays(rand(0, 30));
            $saleNumber = 'SALE-' . date('Ymd', strtotime($saleDate)) . '-' . str_pad($i, 4, '0', STR_PAD_LEFT);
            $paymentMethod = $this->randomPaymentMethod();
            $paymentStatus = rand(0, 10) > 2 ? 'completed' : 'pending'; // 80% completed
            
            // Calculate sale totals
            $subtotal = 0;
            $saleItems = [];
            
            // Add 1-4 items to each sale
            $numItems = rand(1, 4);
            $selectedItems = $inventoryItems->random(min($numItems, $inventoryItems->count()));
            
            foreach ($selectedItems as $item) {
                $quantity = rand(1, 3);
                $unitPrice = $item->selling_price;
                $itemSubtotal = $quantity * $unitPrice;
                $subtotal += $itemSubtotal;
                
                $saleItems[] = [
                    'item' => $item,
                    'quantity' => $quantity,
                    'unit_price' => $unitPrice,
                    'subtotal' => $itemSubtotal,
                ];
            }
            
            $taxAmount = $subtotal * 0.00; // 0% tax for now
            $discountAmount = rand(0, 1) ? 0 : $subtotal * (rand(5, 15) / 100); // Random 0-15% discount
            $totalAmount = $subtotal + $taxAmount - $discountAmount;
            $amountPaid = $paymentStatus === 'completed' ? $totalAmount : 0;
            $changeAmount = $paymentStatus === 'completed' && $paymentMethod === 'cash' ? rand(0, 500) : 0;
            
            if ($changeAmount > 0) {
                $amountPaid += $changeAmount;
            }
            
            // Insert sale
            $saleId = DB::table('sales')->insertGetId([
                'sale_number' => $saleNumber,
                'user_id' => $users[array_rand($users)], // cashier
                'customer_id' => null, // walk-in customer
                'subtotal' => $subtotal,
                'tax_amount' => $taxAmount,
                'discount_amount' => $discountAmount,
                'total_amount' => $totalAmount,
                'payment_method' => $paymentMethod,
                'amount_paid' => $amountPaid,
                'change_amount' => $changeAmount,
                'payment_status' => $paymentStatus,
                'notes' => $this->generateSaleNotes(),
                'receipt_number' => 'RCT-' . date('Ymd', strtotime($saleDate)) . '-' . rand(1000, 9999),
                'sale_date' => $saleDate,
                'created_at' => $saleDate,
                'updated_at' => $saleDate,
            ]);
            
            // Insert sale items
            foreach ($saleItems as $saleItem) {
                $itemTotal = $saleItem['subtotal']; // No item-level discount for now
                $profit = ($saleItem['unit_price'] - $saleItem['item']->buying_price) * $saleItem['quantity'];
                
                DB::table('sale_items')->insert([
                    'sale_id' => $saleId,
                    'inventory_item_id' => $saleItem['item']->id,
                    'item_name' => $saleItem['item']->name, // Snapshot
                    'sku' => $saleItem['item']->sku, // Snapshot
                    'quantity' => $saleItem['quantity'],
                    'unit_price' => $saleItem['unit_price'],
                    'buying_price' => $saleItem['item']->buying_price,
                    'subtotal' => $saleItem['subtotal'],
                    'discount' => 0,
                    'total' => $itemTotal,
                    'profit' => $profit,
                    'created_at' => $saleDate,
                    'updated_at' => $saleDate,
                ]);
                
                // Update inventory quantity
                DB::table('inventory_items')
                    ->where('id', $saleItem['item']->id)
                    ->decrement('quantity', $saleItem['quantity']);
            }
        }
        
        $this->command->info('Successfully seeded 10 sales with items.');
    }
    
    private function generateCustomerName(): string
    {
        $firstNames = ['John', 'Jane', 'Mary', 'James', 'Patricia', 'Michael', 'Linda', 'Robert', 'Barbara', 'David'];
        $lastNames = ['Smith', 'Johnson', 'Williams', 'Brown', 'Jones', 'Garcia', 'Miller', 'Davis', 'Rodriguez', 'Martinez'];
        
        return $firstNames[array_rand($firstNames)] . ' ' . $lastNames[array_rand($lastNames)];
    }
    
    private function randomPaymentMethod(): string
    {
        $methods = ['cash', 'card', 'mobile_money', 'bank_transfer', 'credit'];
        return $methods[array_rand($methods)];
    }
    
    private function generateSaleNotes(): ?string
    {
        $notes = [
            null,
            'Regular customer',
            'First time customer',
            'Requested gift wrapping',
            'Customer was very satisfied',
            'Referred by another client',
        ];
        
        return $notes[array_rand($notes)];
    }
}
