<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class StockMovementsSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $inventoryItems = DB::table('inventory_items')->get();
        $users = DB::table('users')->pluck('user_id')->toArray();
        
        if ($inventoryItems->isEmpty()) {
            $this->command->warn('No inventory items found. Please run InventoryItemsSeeder first.');
            return;
        }
        
        if (empty($users)) {
            $this->command->warn('No users found. Please run UsersSeeder first.');
            return;
        }

        $movements = [];
        
        // Create stock IN movements (purchases/restocks)
        foreach ($inventoryItems->random(min(8, $inventoryItems->count())) as $item) {
            $quantityChange = rand(10, 50);
            $previousQty = max(0, $item->quantity - $quantityChange);
            $newQty = $item->quantity;
            $movementDate = now()->subDays(rand(5, 45));
            
            $movements[] = [
                'inventory_item_id' => $item->id,
                'user_id' => $users[array_rand($users)],
                'sale_id' => null,
                'type' => 'in',
                'quantity' => $quantityChange,
                'previous_quantity' => $previousQty,
                'new_quantity' => $newQty,
                'reference_number' => 'PO-' . date('Ymd') . '-' . rand(1000, 9999),
                'reason' => 'Purchase Order - Regular stock replenishment',
                'created_at' => $movementDate,
                'updated_at' => $movementDate,
            ];
        }
        
        // Create stock OUT movements (damages, returns, etc.)
        foreach ($inventoryItems->where('quantity', '>', 5)->random(min(3, $inventoryItems->where('quantity', '>', 5)->count())) as $item) {
            $quantityChange = rand(1, 5);
            $previousQty = $item->quantity + $quantityChange;
            $newQty = $item->quantity;
            $movementDate = now()->subDays(rand(1, 20));
            
            $movements[] = [
                'inventory_item_id' => $item->id,
                'user_id' => $users[array_rand($users)],
                'sale_id' => null,
                'type' => $this->randomOutType(),
                'quantity' => $quantityChange,
                'previous_quantity' => $previousQty,
                'new_quantity' => $newQty,
                'reference_number' => 'ADJ-' . date('Ymd') . '-' . rand(1000, 9999),
                'reason' => $this->randomOutReason(),
                'created_at' => $movementDate,
                'updated_at' => $movementDate,
            ];
        }
        
        // Create stock ADJUSTMENT movements
        foreach ($inventoryItems->random(min(2, $inventoryItems->count())) as $item) {
            $quantityChange = rand(1, 10);
            $isIncrease = rand(0, 1);
            $previousQty = $isIncrease ? ($item->quantity - $quantityChange) : ($item->quantity + $quantityChange);
            $newQty = $item->quantity;
            $movementDate = now()->subDays(rand(1, 15));
            
            $movements[] = [
                'inventory_item_id' => $item->id,
                'user_id' => $users[array_rand($users)],
                'sale_id' => null,
                'type' => 'adjustment',
                'quantity' => $quantityChange,
                'previous_quantity' => $previousQty,
                'new_quantity' => $newQty,
                'reference_number' => 'STK-' . date('Ymd') . '-' . rand(1000, 9999),
                'reason' => 'Stock Count Adjustment - Inventory reconciliation after physical count',
                'created_at' => $movementDate,
                'updated_at' => $movementDate,
            ];
        }

        foreach ($movements as $movement) {
            DB::table('stock_movements')->insert($movement);
        }
        
        $this->command->info('Successfully seeded ' . count($movements) . ' stock movements.');
    }
    
    private function randomOutType(): string
    {
        $types = ['damage', 'expired', 'return'];
        return $types[array_rand($types)];
    }
    
    private function randomOutReason(): string
    {
        $reasons = [
            'Damaged goods',
            'Expired product',
            'Customer return',
            'Quality issue',
            'Sample/Demo',
        ];
        
        return $reasons[array_rand($reasons)];
    }
}
